#' Generates a random time series with temporal autocorrelation.
#'
#' @description Generates a vector of the same legnth as the \code{time} argument, with a temporal autocorrelation length close to the defined by \code{autocorrelation.length}, and a range within \code{output.min} and \code{output.max}. The output of this function is intended to be used as an input to the function \code{\link{simulatePopulation}}. \strong{Important}: note that the variable \code{time} runs from left to right in \code{\link{simulatePopulation}}, with lower values representing older samples.
#'
#' @usage simulateDriver(
#'   random.seed = 30,
#'   time = 1:10000,
#'   autocorrelation.length = 100,
#'   output.min = 0,
#'   output.max = 100,
#'   rescale = TRUE
#'   )
#'
#' @param random.seed integer, seed to be used by \code{set.seed()}. Default is 50.
#' @param time integer, or numeric vector of integers with constant intervals. If a single integer is provided, a time sequence is generated from 1 to the given integer as \emph{seq(1, time, by = 1)}. Default is 1:10000.
#' @param autocorrelation.length integer, represents the length of the convolution filter to be used to impose a particular temporal structure on the time series. Default is 100, equivalent to a filter composed by a hundred of ones.
#' @param output.min numeric, minimum value of the output time series. Used as input for\code{\link{rescaleVector}}. Default is 0.
#' @param output.max numeric, maximum value of the output time series. Used as input for \code{\link{rescaleVector}}. Default is 100.
#' @param rescale boolean. If FALSE, \code{output.min} and \code{output.max} are ignored, and the original data range provided by \code{rnorm} is preserved. Default is TRUE.
#'
#' @details It is recommended to use \code{time} vectors with a time step of 1 between consecutive values when the output is to be used as input for \code{\link{simulatePopulation}}, which considers annual time-steps while simulating virtual pollen curves. The initial random sequence of numers is generated by \code{rnorm}. Desired temporal autocorrelation are approximate, but deviation becomes higher if \code{autocorrelation.length} is larger than half the length of \code{time}. Consequently, the function limits \code{autocorrelation.length} to \code{length(time)/2}.
#'
#' @author Blas M. Benito  <blasbenito@gmail.com>
#'
#' @return A vector of the same length as \code{time}. Datasets \code{\link{driverA}} and \code{\link{driverB}} are outputs of this function.
#'
#' @seealso \code{\link{rescaleVector}}, \code{\link{driverA}}, \code{\link{driverB}}, \code{\link{set.seed}}
#'
#' @examples
#'
#' x <- simulateDriver(
#'   random.seed = 30,
#'   time = 1:10000,
#'   autocorrelation.length = 100,
#'   output.min = -10,
#'   output.max = 20,
#'   rescale = TRUE
#'   )
#'
#' #plots output
#' plot(x, type = "l")
#'
#' #checks temporal autocorrelation
#' acf(x, lag.max = 300)
#'
#' @export
simulateDriver = function(random.seed = 30,
                          time = 1:10000,
                          autocorrelation.length = 100,
                          output.min = 0,
                          output.max = 100,
                          rescale = TRUE){

  #RANDOM SEED
  #----------------------

  #is integer?
  if(!is.integer(random.seed)){
    random.seed <- floor(random.seed)
    }

  #setting random seed to the specified one
  set.seed(random.seed)


  #TIME
  #----------------------

  #if not a vector, forces "time" as integer and creates vector starting at 0
  if(!is.vector(time)){
    time <- seq(1, floor(time), by = 1)
    if(length(time) == 0){
      error("Time needs to be an integer higher than 1, or a vector of integers.")
    }
  }


  #AUTOCORRELATION LENGTH
  #----------------------

  #limits autocorrelation length to half the length of time if higher
  if (autocorrelation.length > length(time)/2){
    autocorrelation.length <- floor(length(time)/2)
  }


  #GENERATES DRIVER
  #----------------------

  #generates driver (returns time series)
  driver <- filter(rnorm(max(time)),
                  filter = rep(1, autocorrelation.length),
                  circular = TRUE)

  #converts from time series to vector
  driver <- as.vector(driver)

  #rescales time series to [output.min output.max]
  if(rescale == TRUE){
    driver <- rescaleVector(x = driver,
                           new.min = output.min,
                           new.max = output.max)
  }


  return(driver)

}
