#' @export

stoch.bootstrap_cbm <- function(a, S_1, S_2, boot_iter = 500, p = 0.05, seed = NULL) {
  cores <- parallel::detectCores(logical = FALSE)
  cl <- snow::makeCluster(cores[1] - 1)
  doSNOW::registerDoSNOW(cl)

  mu1_est <- a$mu1
  mu2_est <- a$mu2
  sigma1_est <- a$sigma1
  sigma2_est <- a$sigma2
  rho_est <- a$rho
  dt<-a$dt
  nc <- length(S_1)
  if(!is.null(seed)){
    set.seed(seed)
  }
  pb <- progress::progress_bar$new(
    format = " [:bar] :percent eta: :eta",
    total = boot_iter + 1, clear = TRUE, width = 60
  )
  progress <- function(n) {
    pb$tick()
  }

  opts <- list(progress = progress)

  ff <- sim_stochcorr(dt, S_1[1], S_2[1], mu1_est, mu2_est, sigma1_est, sigma2_est, rho_est, iter = boot_iter)

  boot <- list()

  boot <- foreach::foreach(i = 1:boot_iter, .options.snow = opts) %dopar% {
    stoch_cbm_est(ff$S1[i, ], ff$S2[i, ], dt, lambda = a$lambda)
  }
  snow::stopCluster(cl)

  rho_boot <- matrix(0, boot_iter, nc)

  for (i in 1:boot_iter) {
    rho_boot[i, ] <- boot[[i]]$rho
  }

  rho_quantile <- matrix(0, nc, 1)
  for (i in 1:nc) {
    rho_quantile[i, ] <- quantile(1 - cos(acos(rho_boot[, i]) - acos(a$rho[i])), c(1 - (p)))
  }

  lower_rho <- rep(0, nc)
  upper_rho <- rep(0, nc)

  for (i in 1:nc) {
    lower_rho[i] <- a$rho[i] - 1 + rho_quantile[i, ]
    upper_rho[i] <- a$rho[i] + 1 - rho_quantile[i, ]
    if (lower_rho[i] < -1) {
      lower_rho[i] <- -1
    }
    if (upper_rho[i] < -1) {
      upper_rho[i] <- -1
    }
    if (lower_rho[i] > 1) {
      lower_rho[i] <- 1
    }
    if (upper_rho[i] > 1) {
      upper_rho[i] <- 1
    }
    t1 <- min(lower_rho[i], upper_rho[i])
    t2 <- max(lower_rho[i], upper_rho[i])
    lower_rho[i] <- t1
    upper_rho[i] <- t2
  }

  conf_rho <- rbind(lower_rho, upper_rho)
  rownames(conf_rho) <- c("lower bound", "upper bound")
  return(conf_rho)
}

#' @export

circ.bootstrap_cbm <- function(a, theta, boot_iter = 500, p = 0.05, seed = NULL) {
  cores <- parallel::detectCores(logical = FALSE)
  cl <- snow::makeCluster(cores[1] - 1)
  doSNOW::registerDoSNOW(cl)

  sigma_est <- a$sigma_cbm
  dt <- a$dt
  nc <- length(theta)
  if(!is.null(seed)){
    set.seed(seed)
  }
  pb <- progress::progress_bar$new(
    format = " [:bar] :percent eta: :eta",
    total = boot_iter + 1, clear = TRUE, width = 60
  )
  progress <- function(n) {
    pb$tick()
  }

  opts <- list(progress = progress)
  ff <- matrix(0,boot_iter,nc)
  for(i in 1:boot_iter){
    ff[i,] <- rtraj.cbm(nc, theta[1], dt, sigma_est)
  }
  boot <- list()

  boot <- foreach::foreach(i = 1:boot_iter, .options.snow = opts) %dopar% {
    cbm_est(ff[i,], dt, lambda = a$lambda)
  }
  snow::stopCluster(cl)

  param <- matrix(0, boot_iter, 1)

  for (i in 1:boot_iter) {
    param[i, ] <- c(boot[[i]]$sigma_cbm)
  }

  sigma_quantile <- quantile(param[,1]-sigma_est,c(1-p/2,p/2))

  lower_conf <- c(sigma_est-sigma_quantile[1])
  upper_conf <- c(sigma_est-sigma_quantile[2])

  conf <- rbind(lower_conf, upper_conf)
  rownames(conf) <- c("lower bound", "upper bound")
  colnames(conf) <- c("sigma")
  return(conf)
}

