% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vMF.R
\name{vMF}
\alias{vMF}
\alias{d_vMF}
\alias{c_vMF}
\alias{r_vMF}
\alias{g_vMF}
\alias{r_g_vMF}
\title{von Mises--Fisher distribution}
\usage{
d_vMF(x, mu, kappa, log = FALSE)

c_vMF(p, kappa, log = FALSE)

r_vMF(n, mu, kappa)

g_vMF(t, p, kappa, scaled = TRUE, log = FALSE)

r_g_vMF(n, p, kappa)
}
\arguments{
\item{x}{locations in \eqn{S^{p-1}} to evaluate the density. Either a
matrix of size \code{c(nx, p)} or a vector of length \code{p}. Normalized
internally if required (with a \code{warning} message).}

\item{mu}{the directional mean \eqn{\boldsymbol{\mu}}{\mu} of the vMF.
A unit-norm vector of length \code{p}.}

\item{kappa}{concentration parameter \eqn{\kappa} of the vMF.
A nonnegative scalar. Can be a vector for \code{c_vMF}.}

\item{log}{flag to indicate if the logarithm of the density (or the
normalizing constant) is to be computed.}

\item{p}{dimension of the ambient space \eqn{R^p} that contains
\eqn{S^{p-1}}. A positive integer.}

\item{n}{sample size, a positive integer.}

\item{t}{a vector with values in \eqn{[-1, 1]}.}

\item{scaled}{whether to scale the angular function by the von Mises--Fisher
normalizing constant. Defaults to \code{TRUE}.}
}
\value{
Depending on the function:
\itemize{
\item \code{d_vMF}: a vector of length \code{nx} or \code{1} with the
evaluated density at \code{x}.
\item \code{r_vMF}: a matrix of size \code{c(n, p)} with the random sample.
\item \code{c_vMF}: the normalizing constant.
\item \code{g_vMF}: a vector of size \code{length(t)} with the evaluated
angular function.
\item \code{r_g_vMF}: a vector of length \code{n} containing simulated
values from the cosines density associated to the angular function.
}
}
\description{
Density and simulation of the von Mises--Fisher (vMF)
distribution on
\eqn{S^{p-1}:=\{\mathbf{x}\in R^p:||\mathbf{x}||=1\}}{
S^{p-1} := \{x \in R^p : ||x|| = 1\}}, \eqn{p\ge 1}. The density at
\eqn{\mathbf{x} \in S^{p-1}}{x \in S^{p-1}} is given by
\deqn{c^{\mathrm{vMF}}_{p,\kappa}
e^{\kappa\mathbf{x}' \boldsymbol{\mu}}
\quad\mathrm{with}\quad c^{\mathrm{vMF}}_{p,\kappa}:=
\kappa^{(p-2)/2}/((2\pi)^{p/2} I_{(p-2)/2}(\kappa))}{
c^{vMF}_{p,\kappa} e^{\kappa x' \mu} with
c^{vMF}_{p,\kappa} :=
\kappa^{(p-2)/2}/((2\pi)^{p/2} I_{(p-2)/2}(\kappa))}
where \eqn{\boldsymbol{\mu}\in S^{p-1}}{\mu \in S^{p-1}} is the directional
mean, \eqn{\kappa\ge 0} is the concentration parameter about
\eqn{\boldsymbol{\mu}}{\mu}, and \eqn{I_\nu} is the order-\eqn{\nu}
modified Bessel function of the first kind.

The angular function of the vMF is \eqn{g(t) := e^{\kappa t}}. The
associated \emph{cosines} density is
\eqn{\tilde g(v):= \omega_{p-1} c^{\mathrm{vMF}}_{p,\kappa}
g(v) (1 - v^2)^{(p-3)/2}}{
\tilde g(v):= \omega_{p-1} c^{vMF}_{p,\kappa} g(v)(1 - v^2)^{(p-3)/2}},
where \eqn{\omega_{p-1}} is the surface area of \eqn{S^{p-2}}.
}
\details{
\code{r_g_vMF} implements algorithm VM in Wood (1994). \code{c_vMF} is
vectorized on \code{p} and \code{kappa}.
}
\examples{
# Simulation and density evaluation for p = 2
mu <- c(0, 1)
kappa <- 2
n <- 1e3
x <- r_vMF(n = n, mu = mu, kappa = kappa)
dens <- d_vMF(x = x, mu = mu, kappa = kappa)
col <- viridisLite::viridis(n)
r <- runif(n, 0.95, 1.05) # Radius perturbation to improve visualization
plot(r * x, pch = 16, col = col[rank(dens)])

# Simulation and density evaluation for p = 3
mu <- c(0, 0, 1)
kappa <- 2
x <- r_vMF(n = n, mu = mu, kappa = kappa)
dens <- d_vMF(x = x, mu = mu, kappa = kappa)
scatterplot3d::scatterplot3d(x, xlim = c(-1, 1), ylim = c(-1, 1),
                             zlim = c(-1, 1), color = col[rank(dens)],
                             pch = 16, xlab = "", ylab = "", zlab = "",
                             angle = 20)

# Cosines density
g_tilde <- function(t, p, kappa) {
  exp(w_p(p = p - 1, log = TRUE) +
        g_vMF(t = t, p = p, kappa = kappa, scaled = TRUE, log = TRUE) +
        ((p - 3) / 2) * log(1 - t^2))
}

# Simulated data from the cosines density
n <- 1e3
p <- 3
kappa <- 2
hist(r_g_vMF(n = n, p = p, kappa = kappa), breaks = seq(-1, 1, l = 20),
     probability = TRUE, main = "Simulated data from g_vMF", xlab = "t")
t <- seq(-1, 1, by = 0.01)
lines(t, g_tilde(t = t, p = p, kappa = kappa))

# Cosine density as a function of the dimension
M <- 100
col <- viridisLite::viridis(M)
plot(t, g_tilde(t = t, p = 2, kappa = kappa), col = col[2], type = "l",
     ylab = "Density")
for (p in 3:M) {
  lines(t, g_tilde(t = t, p = p, kappa = kappa), col = col[p])
}
}
\references{
Wood, A. T. A. (1994) Simulation of the von Mises Fisher distribution.
\emph{Commun. Stat. Simulat.}, 23(1):157--164.
\doi{10.1080/03610919408813161}
}
\seealso{
\code{\link{tangent-vMF}}.
}
\author{
Eduardo García-Portugués, Davy Paindaveine, and Thomas Verdebout.
}
