% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/h.cv.R
\name{h.cv}
\alias{h.cv}
\alias{h.cv.bin.data}
\alias{h.cv.bin.den}
\alias{h.cv.svar.bin}
\alias{hcv.data}
\title{Cross-validation methods for bandwidth selection}
\usage{
h.cv(bin, ...)

\method{h.cv}{bin.data}(
  bin,
  objective = c("CV", "GCV", "MASE"),
  h.start = NULL,
  h.lower = NULL,
  h.upper = NULL,
  degree = 1,
  ncv = ifelse(objective == "CV", 2, 0),
  cov.bin = NULL,
  DEalgorithm = FALSE,
  warn = TRUE,
  tol.mask = npsp.tolerance(2),
  ...
)

\method{h.cv}{bin.den}(
  bin,
  h.start = NULL,
  h.lower = NULL,
  h.upper = NULL,
  degree = 1,
  ncv = 2,
  DEalgorithm = FALSE,
  ...
)

\method{h.cv}{svar.bin}(
  bin,
  loss = c("MRSE", "MRAE", "MSE", "MAE"),
  h.start = NULL,
  h.lower = NULL,
  h.upper = NULL,
  degree = 1,
  ncv = 1,
  DEalgorithm = FALSE,
  warn = FALSE,
  ...
)

hcv.data(
  bin,
  objective = c("CV", "GCV", "MASE"),
  h.start = NULL,
  h.lower = NULL,
  h.upper = NULL,
  degree = 1,
  ncv = ifelse(objective == "CV", 1, 0),
  cov.dat = NULL,
  DEalgorithm = FALSE,
  warn = TRUE,
  ...
)
}
\arguments{
\item{bin}{object used to select a method (binned data, binned density or binned semivariogram).}

\item{...}{further arguments passed to or from other methods
(e.g. parameters of the optimization routine).}

\item{objective}{character; optimal criterion to be used ("CV", "GCV" or "MASE").}

\item{h.start}{vector; initial values for the parameters (diagonal elements) to be optimized over.
If \code{DEalgorithm == FALSE} (otherwise not used), defaults to \code{(3 + ncv) * lag},
where \code{lag = bin$grid$lag}.}

\item{h.lower}{vector; lower bounds on each parameter (diagonal elements) to be optimized.
Defaults to \code{(1.5 + ncv) * bin$grid$lag}.}

\item{h.upper}{vector; upper bounds on each parameter (diagonal elements) to be optimized.
Defaults to \code{1.5 * dim(bin) * bin$grid$lag}.}

\item{degree}{degree of the local polynomial used. Defaults to 1 (local linear estimation).}

\item{ncv}{integer; determines the number of cells leaved out in each dimension.
(0 to GCV considering all the data, \eqn{>0} to traditional or modified cross-validation).
See "Details" bellow.}

\item{cov.bin}{(optional) covariance matrix of the binned data or semivariogram model
(\code{\link{svarmod}}-class) of the (unbinned) data. Defaults to the identity matrix.}

\item{DEalgorithm}{logical; if \code{TRUE}, the differential evolution optimization algorithm 
in package \pkg{DEoptim} is used.}

\item{warn}{logical; sets the handling of warning messages
(normally due to the lack of data in some neighborhoods).
If \code{FALSE} all warnings are ignored.}

\item{tol.mask}{tolerance used in the aproximations. Defaults to \code{\link{npsp.tolerance}(2)}.}

\item{loss}{character; CV error. See "Details" bellow.}

\item{cov.dat}{covariance matrix of the data or semivariogram model
(of class extending \code{\link{svarmod}}). Defaults to the identity matrix
(uncorrelated data).}
}
\value{
Returns a list containing the following 3 components:
\item{h}{the best (diagonal) bandwidth matrix found.} 
\item{value}{the value of the objective function corresponding to \code{h}.}
\item{objective}{the criterion used.}
}
\description{
Selects the bandwidth of a local polynomial kernel (regression, density or
variogram) estimator using (standard or modified) CV, GCV or MASE criteria.
}
\details{
Currently, only diagonal bandwidths are supported.

\code{h.cv} methods use binning approximations to the objective function values 
(in almost all cases, an averaged squared error). 
If \code{ncv > 0}, estimates are computed by leaving out binning cells with indexes within 
the intervals \eqn{[x_i - ncv + 1, x_i + ncv - 1]}, at each dimension i, where \eqn{x} 
denotes the index of the estimation location. \eqn{ncv = 1} corresponds with
traditional cross-validation and \eqn{ncv > 1} with modified CV 
(it may be appropriate for dependent data; see e.g. Chu and Marron, 1991, for the one dimensional case). 
Setting \code{ncv >= 2} would be recommended for sparse data (as linear binning is used).
For standard GCV, set \code{ncv = 0} (the whole data would be used).
For theoretical MASE, set \code{bin = binning(x, y = trend.teor)}, \code{cov = cov.teor} and \code{ncv = 0}.

If \code{DEalgorithm == FALSE}, the \code{"L-BFGS-B"} method in \code{\link{optim}} is used.

The different options for the argument \code{loss} in \code{h.cv.svar.bin()} define the CV error 
considered in semivariogram estimation:
\describe{
 \item{\code{"MSE"}}{Mean squared error}
 \item{\code{"MRSE"}}{Mean relative squared error}
 \item{\code{"MAE"}}{Mean absolute error}
 \item{\code{"MRAE"}}{Mean relative absolute error}
}

\code{hcv.data} evaluates the objective function at the original data
(combining a binning approximation to the nonparametric estimates with a linear interpolation),
this can be very slow (and memory demanding; consider using \code{h.cv} instead).
If \code{ncv > 1} (modified CV), a similar algorithm to that in \code{h.cv} is used,
estimates are computed by leaving out binning cells with indexes within
the intervals \eqn{[x_i - ncv + 1, x_i + ncv - 1]}.
}
\examples{
# Trend estimation
bin <- binning(earthquakes[, c("lon", "lat")], earthquakes$mag)
hcv <- h.cv(bin, ncv = 2)
lp <- locpol(bin, h = hcv$h)
# Alternatively, `locpolhcv()` could be called instead of the previous code. 

simage(lp, main = 'Smoothed magnitude')
contour(lp, add = TRUE)
with(earthquakes, points(lon, lat, pch = 20))

# Density estimation
hden <- h.cv(as.bin.den(bin))
den <- np.den(bin, h = hden$h)

plot(den, main = 'Estimated log(density)')
}
\references{
Chu, C.K. and Marron, J.S. (1991) Comparison of Two Bandwidth Selectors
  with Dependent Errors. \emph{The Annals of Statistics}, \bold{19}, 1906-1918.

Francisco-Fernandez M. and Opsomer J.D. (2005) Smoothing parameter selection
 methods for nonparametric regression with spatially correlated errors. 
 \emph{Canadian Journal of Statistics}, \bold{33}, 539-558.
}
\seealso{
\code{\link{locpol}}, \code{\link{locpolhcv}}, \code{\link{binning}}, 
\code{\link{np.den}}, \code{\link{np.svar}}.
}
