# WARNING - Generated by {fusen} from dev/flat_package.Rmd: do not edit by hand

#' Graphical representation of simulated data
#' 
#'
#' 
#' @importFrom dplyr bind_rows
#' @importFrom ggplot2 ggplot aes geom_point geom_line 
#' theme_bw scale_colour_brewer
#' 
#' @param simu list, output of [funStatTest::simul_data()]
#' 
#' @return the ggplot2 graph of simulated tajectories.
#' 
#' @seealso [funStatTest::simul_data()]
#' 
#' @inherit funStatTest_authors author
#' 
#' @export
#'
#' @examples
#' # constant delta
#' simu_data <- simul_data(
#'     n_point = 100, n_obs1 = 50, n_obs2 = 75, c_val = 5, 
#'     delta_shape = "constant", distrib = "normal"
#' )
#' plot_simu(simu_data)
#' # linear delta
#' simu_data <- simul_data(
#'     n_point = 100, n_obs1 = 50, n_obs2 = 75, c_val = 5, 
#'     delta_shape = "linear", distrib = "normal"
#' )
#' plot_simu(simu_data)
#' # quadratic delta
#' simu_data <- simul_data(
#'     n_point = 100, n_obs1 = 50, n_obs2 = 75, c_val = 5, 
#'     delta_shape = "quadratic", distrib = "normal"
#' )
#' plot_simu(simu_data)
plot_simu <- function(simu) {
    
    # format data
    data2plot <- bind_rows(
        prepare_data(simu$mat_sample1, sample_id = 1),
        prepare_data(simu$mat_sample2, sample_id = 2)
    )
    
    # graph
    graph <- ggplot(
        data2plot, aes(x=point, y=value, col=sample, 
                       group=interaction(sample, trajectory))) +
        geom_line(alpha = 0.5) +
        scale_colour_brewer(palette="Dark2") +
        theme_bw()
    
    return(graph)
}

#' Simulation of trajectories from two samples diverging by a delta function
#' 
#' @description
#' Simulate `n_obs1` trajectories of length `n_point` in the first sample and 
#' `n_obs2` trajectories of length `n_point` in the second sample.
#' 
#'
#' 
#' @importFrom tibble lst
#' 
#' @inheritParams simul_traj
#' @param n_obs1 integer value, number of trajectories in the first sample.
#' @param n_obs2 integer value, number of trajectories in the second sample.
#' @param c_val numeric value, level of divergence between the two samples.
#' @param delta_shape character string, shape of the divergence between the 
#' two samples, among `"constant"`, `"linear"`, `"quadratic"`.
#'
#' @return A list with the following elements
#' - `mat_sample1`: numeric matrix of dimension `n_point x n_obs1` containing 
#' `n_obs1` trajectories (in columns) of size `n_point` (in rows) 
#' corresponding to sample 1.
#' - `mat_sample2`: numeric matrix of dimension `n_point x n_obs2` containing 
#' `n_obs2` trajectories (in columns) of size `n_point` (in rows) 
#' corresponding to sample 2.
#' 
#' @seealso [funStatTest::plot_simu()], [funStatTest::simul_traj()]
#' 
#' @inherit funStatTest_authors author
#' 
#' @references Zaineb Smida, Lionel Cucala, Ali Gannoun & Ghislain Durif (2022) 
#' A median test for functional data, Journal of Nonparametric Statistics, 
#' 34:2, 520-553,  
#' \doi{10.1080/10485252.2022.2064997}, 
#' [hal-03658578](https://hal.science/hal-03658578)
#' 
#' @export
#'
#' @examples
#' simu_data <- simul_data(
#'     n_point = 100, n_obs1 = 50, n_obs2 = 75, c_val = 10, 
#'     delta_shape = "constant", distrib = "normal"
#' )
#' str(simu_data)
simul_data <- function(
        n_point, n_obs1, n_obs2, c_val = 0, delta_shape = "constant", 
        distrib = "normal", max_iter = 10000) {
    
    assert_count(n_point, positive = TRUE)
    assert_count(n_obs1, positive = TRUE)
    assert_count(n_obs2, positive = TRUE)
    assert_numeric(c_val, len = 1)
    assert_choice(
        delta_shape, c("constant", "linear", "quadratic"), null.ok = FALSE)
    assert_count(max_iter, positive = TRUE)
    
    # init
    t_vec <- (0:(n_point-1))/(n_point-1)
    
    # delta values
    delta <- switch (
        delta_shape,
        constant = c_val,
        linear = c_val * t_vec,
        quadratic = c_val * t_vec * (1 - t_vec)
    )
    
    # MatX
    mat_sample1 <- matrix(0, n_point, n_obs1)
    for(i in 1:n_obs1)
        mat_sample1[,i] <- simul_traj(n_point, distrib, max_iter) + delta
    
    # MatY
    mat_sample2 <- matrix(0, n_point, n_obs2)
    for(j in 1:n_obs2)
        mat_sample2[,j] <- simul_traj(n_point, distrib, max_iter)
    
    # output
    return(lst(mat_sample1, mat_sample2, c_val, distrib, delta_shape))
}

#' Single trajectory simulation
#' 
#'
#' 
#' @description
#' Simulate a trajectory of length `n_point` using a random generator 
#' associated to different probability distribution.
#'  
#' @param n_point integer value, number of points in the trajectory.
#' @param distrib character string, type of probability distribution used to 
#' simulate the data among `"normal"`, `"cauchy"`, `"dexp"`, `"student"`.
#' @param max_iter integer, maximum number of iteration for the iterative 
#' simulation process.
#' 
#' @return
#' Vector of size `n_point` with the trajectory values.
#' 
#' @importFrom stats rnorm rcauchy rt
#' @importFrom distr r DExp
#' 
#' @inherit funStatTest_authors author
#' 
#' @references Zaineb Smida, Lionel Cucala, Ali Gannoun & Ghislain Durif (2022) 
#' A median test for functional data, Journal of Nonparametric Statistics, 
#' 34:2, 520-553,  
#' \doi{10.1080/10485252.2022.2064997}, 
#' [hal-03658578](https://hal.science/hal-03658578)
#' 
#' @export
#' 
#' @seealso [funStatTest::simul_data()]
#' 
#' @examples
#' simu_vec <- simul_traj(100)
#' plot(simu_vec, xlab = "point", ylab = "value")
simul_traj <- function(n_point, distrib = "normal", max_iter = 10000) {
    
    assert_count(n_point, positive = TRUE)
    assert_choice(
        distrib, c("normal", "cauchy", "dexp", "student"), null.ok = FALSE)
    assert_count(max_iter, positive = TRUE)
    
    # init
    vect <- (0:(n_point-1))/(n_point-1)
    vecY <- numeric(n_point)
    n_iter <- 0
    
    # iter
    for(k in 1:max_iter) {
        
        sigma <- 1/((k-0.5)*pi)
        
        y <- switch(
            distrib,
            normal = rnorm(1, mean = 0, sd = sigma),
            cauchy = sigma * rcauchy(1, location = 0, scale = 1),
            dexp = sigma * r(DExp(rate=1))(1),
            student = sigma*rt(1, df = 5)
        )
        
        vecyphi <- sqrt(2)*sin(vect/sigma)
        
        # store previous value of vecY
        exvecY <- vecY
        # update
        vecY <- vecY + y * vecyphi
        
        # stopping criterion (depending on the process convergence)
        if(sum((vecY-exvecY)^2)/sum((vecY)^2) < 0.001) break
        # record current iteration
        n_iter <- k
    }
    
    # check convergence
    if(n_iter == max_iter) warning("Simulation process did not converge.")
    
    # output
    return(vecY)
}
