% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trueskillthroughtime.R
\docType{class}
\name{History-class}
\alias{History-class}
\alias{History}
\title{History}
\arguments{
\item{composition}{A list of list of player's names (id). Each position of the 
list is a list that represents the teams of a game, so the latter must contain
vectors of names representing the composition of each team in that game.}

\item{results}{A list of numeric vectors, representing the outcome of each 
game. It must have the same 
length as the \code{composition} list or be an empty list. The default value is an empty list.
When the list is empty, the outcomes of the games are inferred by the order of
the teams in the \code{composition} list: the teams appearing firstly in the list 
defeat those appearing later (no ties).
When the list is not empty, each vector of the list must represents the score of each team in the game. The team with the highest score is 
the winner, and the teams with the same score are tied.}

\item{times}{A numeric vector, the timestamp of each game. It must have the 
same length as the \code{composition} list or be an empty list. The default value 
is an empty list. When the list is empty, all players' games are 
separated by a single timestamp, so a single dynamic uncertainty \code{gamma} will be added between games.
When the list is not empty, the amount of dynamic uncertainty will depend on the difference (measured in timestamps) that each player has between games.
In addition, the order of the timestamps determines the reading order of the \code{composition} and \code{results} lists.
If a player appears more than once in the same timestamp, no dynamic uncertainty will be added between those games.}

\item{priors}{A hash object, a dictionary of \code{Player} objects indexed by the 
players' name (id). Used to create players with special values. The default
value is an empty hash. In this case, one \code{Player} object for each unique 
name in the \code{composition} list is automatically initialized using the values 
of the parameters \code{mu}, \code{sigma}, \code{beta}, and \code{gamma}.
The names that appear in the hash are the only ones that will be initialized 
with special values.}

\item{mu}{A number, the prior mean. The deafult value is: \code{MU = 0}.}

\item{sigma}{A number, the prior standar deviation. The deafult value is: 
\code{SIGMA = 6}.}

\item{beta}{A number, the standard deviation of the performance. The default 
value is: \code{BETA = 1}. The parameter \code{beta} acts as the scale of the 
estimates. A real difference of one \code{beta} between two skills is equivalent 
to 76\% probability of winning.}

\item{gamma}{A number, the amount of uncertainty (standar deviation) added to 
the estimates between events. The default value is: \code{GAMMA = 0.03}.}

\item{p_draw}{A number, the probability of a draw. The default value is 
\code{P_DRAW = 0}. A rule of thumb states that the probability of a draw must be 
initialized with the observed frequency of draws. If in doubt, it is a candidate 
parameter to be optimized or integrated by the sum rule. It is used to compute 
the prior probability of the observed result, so its value may affect an 
eventual model selection task.}

\item{epsilon}{A number, the convergence threshold. Used to stop the convergence procedure. The default value is \code{EPSILON = 1e-6}.}

\item{iterations}{A number, the maximum number of iterations for convergence. Used to stop the convergence procedure. The default value is \code{ITERATIONS = 30}.}
}
\value{
History object
}
\description{
History class
}
\section{Fields}{

\describe{
\item{\code{size}}{A number, the amount of games.}

\item{\code{batches}}{A vector of \code{Batch} objects. Where the games that occur at the same timestamp live.}

\item{\code{agents}}{A hash, a dictionary indexed by the players' name (id).}

\item{\code{time}}{A boolean, indicating whether the history was initialized with timestamps or not.}

\item{\code{mu}}{A number, the default prior mean in this particular \code{History} object}

\item{\code{sigma}}{A number, the default prior standard deviation in this particular \code{History} object}

\item{\code{beta}}{A number, the default standar deviation of the performance in this particular \code{History} object}

\item{\code{gamma}}{A number, the default dynamic uncertainty in this particular \code{History} object}

\item{\code{p_draw}}{A number, the probability of a draw in this particular \code{History} object}

\item{\code{h_epsilon}}{A number, the convergence threshold in this particular \code{History} object}

\item{\code{h_iterations}}{A number, the maximum number of iterations for convergence in this particular \code{History} object}
}}

\section{Methods}{

\describe{
\item{\code{convergence(epsilon = NA, iterations = NA, verbose = TRUE)}}{}

\item{\code{initialize(
  composition,
  results = list(),
  times = c(),
  priors = hash(),
  mu = MU,
  sigma = SIGMA,
  beta = BETA,
  gamma = GAMMA,
  p_draw = P_DRAW,
  epsilon = EPSILON,
  iterations = ITERATIONS
)}}{}

\item{\code{learning_curves()}}{}

\item{\code{log_evidence()}}{}
}}

\examples{
c1 = list(c("a"),c("b"))
c2 = list(c("b"),c("c"))
c3 = list(c("c"),c("a"))
composition = list(c1,c2,c3)
h = History(composition, gamma=0.0)

trueskill_learning_curves = h$learning_curves()
ts_a = trueskill_learning_curves[["a"]]
ts_a[[1]]$N; ts_a[[2]]$N
ts_a[[1]]$t; ts_a[[2]]$t
h$convergence()
trueskillThrougTime_learning_curves = h$learning_curves()
ttt_a = trueskillThrougTime_learning_curves[["a"]]
ttt_a[[1]]$N; ttt_a[[2]]$N
ttt_a[[1]]$t; ttt_a[[2]]$t

\dontrun{
# Synthetic example
library(hash)
N = 100
skill <- function(experience, middle, maximum, slope){
return(maximum/(1+exp(slope*(-experience+middle)))) }
target = skill(seq(N), N/2, 2, 0.075)
opponents = rnorm(N,target,0.5)
composition = list(); results = list(); times = c(); priors = hash()
for(i in seq(N)){composition[[i]] = list(c("a"), c(toString(i)))}
for(i in
seq(N)){results[[i]]=if(rnorm(1,target[i])>rnorm(1,opponents[i])){c(1,0)}else{c(0,1)}}
for(i in seq(N)){times = c(times,i)}
for(i in seq(N)){priors[[toString(i)]] = Player(Gaussian(opponents[i],0.2))}
h = History(composition, results, times, priors, gamma=0.1)
h$convergence(); lc_a = h$learning_curves()$a; mu = c()
for(tp in lc_a){mu = c(mu,tp[[2]]@mu)}
plot(target)
lines(mu)

# Plotting learning curves

# First solve your own example. Here is a dummy one.
agents <- c("a", "b", "c", "d", "e")
composition <- list()
for (i in 1:500) {
 who = sample(agents, 2)
 composition[[i]] <- list(list(who[1]), list(who[2]))
}
h <- History(composition = composition, gamma = 0.03, sigma = 1.0)
h$convergence(iterations=6)

# Then plot some learning curves
lc <- h$learning_curves()
colors <- c(rgb(0.2,0.2,0.8), rgb(0.2,0.8,0.2), rgb(0.8,0.2,0.2))
colors_alpha <- c(rgb(0.2,0.2,0.8,0.2), rgb(0.2,0.8,0.2,0.2), rgb(0.8,0.2,0.2,0.2))
plot(0,0, xlim = c(0, 500), ylim = c(-1, 1), xlab = "t", ylab = "skill", type = "n")
for (i in 1:3) {
  agent <- agents[i]
  t <- c(); mu <- c(); sigma <- c()
  for(x in lc[[agent]]){
    t <- c(t, x$t )
    mu <- c(mu, x$N@mu)
    sigma <- c(sigma, x$N@sigma)
  }
  lines(t, mu, col = colors[i], lwd = 2, type = "l")
  polygon(c(t, rev(t)), c(mu + sigma, rev(mu - sigma)), col = colors_alpha[i], border = NA)
}
legend("topright", legend = agents[1:3], col = colors, lwd = 2)

}
}
