#' @title Subgame of a coalition
#' @description Given a game and a coalition, this function returns the characteristic function of the subgame of the given coalition.
#' @param v A characteristic function, as a vector.
#' @param S The position of a coalition, as an integer.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} and \code{S} are introduced according to binary order instead of lexicographic order.
#' @return  The characteristic function of the subgame of the given coalition, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Given \eqn{v\in G^N}, the subgame of coalition \eqn{S \in 2^{N}} is defined by
#' \eqn{v_S(T)=v(T)} for all \eqn{T\in 2^S}.
#' @examples
#' v <- c(0, 0, 0, 0, 2, 3, 4, 5, 6, 9, 1, 15, 20, 30, 100)
#' S <- 13
#' subgame(v, S)
#' n <- 4
#' for (i in 1 : (2^n-1)) {
#'   cat("[", i, "]", paste(subgame(v,i)), "\n")
#' }
#' subgame(v,15)==v
#' @export

subgame <- function(v, S, binary = FALSE) { # Principio de la función

  ################################
  ### Comprobación datos entrada###
  ################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if (S > nC) {
    stop("'S' must not be greater than the length of 'v'.")
  }
  if (binary == FALSE) { # Si v y S se introdujeron en lexicográfico, los pasamos a binario.
    v <- lex2bin(v)
    S <- codelex2bin(n, S)
  }

  ################################
  ##### Creación de variables######
  ################################

  vS <- c()

  ################################
  ###### Cuerpo de la función######
  ################################

  # Jugadores pertenecientes a la coalición S
  J <- floor(log2(S)) + 1
  Sjug <- c() # Inicializo la coalición S

  for (ii in 1:J)
  {
    if (S %% 2 == 1) {
      Sjug <- c(Sjug, ii)
    }
    S <- S %/% 2
  }

  nS <- length(Sjug)
  Spos <- 2^{
    Sjug - 1
  } # Posición que ocupan los jugadores de S en el juego v.

  for (R in 1:(2^nS - 1))
  { # Inicializo coalición R para calcular los valores en vS
    num <- R # Calculo los jugadores que pertenecen a R.
    J <- floor(log2(num)) + 1
    Rjug <- c() #

    for (ii in 1:J)
    {
      if (num %% 2 == 1) {
        Rjug <- c(Rjug, ii)
      }
      num <- num %/% 2
    }

    vS[R] <- v[sum(Spos[Rjug])] # Posición que ocupan en v los jugadores de R.
  }

  ################################
  ###### Salidas de la función#####
  ################################

  if (binary == FALSE) { # Devolvemos el subjuego en orden lexicográfico.
    return(bin2lex(vS))
  } else { # Devolvemos el subjuego en orden binario.
    return(vS)
  }
} # Fin de la función
