#' @title Least core
#' @description Given a game, this function computes its least core.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return This function returns four outputs:
#'
#' \item{\code{t}}{The excess value that defines the least core.}
#'
#' \item{\code{sat}}{The positions (binary order positions if \code{binary=TRUE}; lexicographic order positions otherwise) of the saturated coalitions, as a vector.}
#'
#' \item{\code{x}}{A least core allocation, as a vector.}
#'
#' \item{\code{vt}}{The game whose core is the least core of \code{v}, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.}
#' @details Given a game \eqn{v\in G^N} and a number \eqn{\varepsilon \in \mathbb{R}}, the \eqn{\varepsilon}-core of \eqn{v} is defined as
#' \deqn{C_{\varepsilon}(v)= \{ x\in \mathbb{R}^n : x(N)=v(N) \text{ and } x(S) \ge v(S)-\varepsilon \ \forall S \in 2^N \setminus \{\emptyset,N\} \},}
#' where \eqn{x(S)=\sum_{i\in S} x_i}.
#' The least core of \eqn{v} is defined as the intersection of all non-empty \eqn{\varepsilon}-cores of \eqn{v}:
#' \deqn{LC(v) = \{ \bigcap_{\varepsilon \in \mathbb{R} \ : \ C_{\varepsilon}(v) \neq \emptyset} C_{\varepsilon}(v) \}.}
#'
#' The implementation of this function is based on the algorithm presented in Derks and Kuipers (1997) and on the MATLAB package WCGT2005 by J. Derks.
#' @examples
#' v <- c(0,0,0,0,10,40,30,60,10,20,90,90,90,130,160)
#' ( vt <- leastcore(v)$vt )
#' # Plotting the core and the least core of v:
#' plotcoresets(games = rbind(v,vt), imputations = FALSE)
#'
#' # What if the game is a cost game?
#' cost.v <- c(2,2,2,3,4,4,5) # characteristic function of the cost game
#' -leastcore(-cost.v)$t # the excess value that defines the least core of cost.v
#' leastcore(-cost.v)$sat # the saturated coalitions
#' -leastcore(-cost.v)$x # a least core allocation
#' -leastcore(-cost.v)$vt # the cost game whose core is the least core of cost.v
#' @references Derks, J. & Kuipers, J. (1997). Implementing the simplex method for computing the prenucleolus of transferable utility games.
#' @references Software by J. Derks (Copyright 2005 Universiteit Maastricht, dept. of Mathematics), available in package \emph{MatTuGames},
#'
#'             \url{https://www.shorturl.at/i6aTF}.
#' @seealso \link{excesses}, \link{nucleoluspcvalue}, \link{nucleolusvalue}, \link{prenucleolusvalue}
#' @export

leastcore <- function(v, binary = FALSE, tol = 100*.Machine$double.eps) {

  nC <- length(v) # número de coaliciones (nC)
  n <- log(nC + 1, base = 2) # número de jugadores
  n <- as.integer(n) # asegurar que n es un entero
  # por qué pusieron la línea anterior en esta función y no en otras?

  if(n>floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ############################
  # Queremos resolver el problema de optimización:
  # min{t : Fx=v(N), Ax+ut\geq v(S) }

  # Inicializamos el procedimiento
  # Asignacion inicial: x = solucion igualitaria
  x <- rep(v[nC] / n, n)

  # Coalicion que se satura
  Fam <- nC  # nC=[1 ... 1]=x1+...+xn

  # Las otras coaliciones: f == A
  f <- AfromFam(Fam, n, tol) # (Todas las coaliciones excepto nC)


  # Sea t0=max{v(S)-x(S): S=1,...,nC-1}=-min{x(S)-v(S):S=1,...,nC-1}
  # y sea j0 el menor índice donde se alcanza t0. Luego j0 determina una
  # coalición A0.
  # Estos valores pueden calcularse con pasoptimo(x,y,t,v,f)
  # para x,y=0,t=0,v,f
  # Inicializacion de variables
  y <- rep(0, n)
  t <- 0

  # Calculo del primer paso
  res <- pasoptimo(x, y, t, v, f, tol)
  t1 <- res$lambda
  A <- res$jj

  # Actualizamos el exceso, que se alcanza en la coalición A
  t <- -t1
  # Otros valores iniciales
  control <- 0
  step = 1
  while (control == 0) {
    # Resolvemos el sistema zF + uA(j0) = 0, u1 = 1
    # Utilizamos la función [y,flag,sat]=sistemaFamA(F,j0,n)
    sistema <- sistemaFamA(Fam, A, n, tol)
    y <- sistema$y
    exit <- sistema$exit
    sat <- sistema$sat

    if (exit == -1) {
      control <- 1  # Si alcanzamos el óptimo
      # sat ya está calculado
    } else if (exit == 0) { # Si no alcanzamos el optimo porque exit=0
      # Calculamos el nuevo paso
      paso <- pasoptimo(x, y, t, v, f, tol)
      lambda <- paso$lambda
      hatj <- paso$jj

      # Nueva solucion factible
      x <- x + lambda * y
      t <- t - lambda

      # Nueva matriz A
      A <- c(A, hatj)
    }
    # Si no alcanzamos el optimo y exit>=1
    else { # # Calculamos el nuevo paso
      res <- pasoptimo(x, y, t, v, f, tol)
      lambda <- res$lambda
      hatj <- res$hatj

      # Nueva solución factible
      x <- x + lambda * y
      t <- t - lambda
      # Nueva matriz A, cambiamos la fila exit por la nueva donde se alcanza el nuevo paso
      A[exit] <- hatj
    } # fin del bucle switch
    #readline(prompt = "Presiona [Enter] para continuar...")
    #stop("para")
    step = step +1
  } # fin del bucle while
  # LEASTCORE allocation x found along with vector sat
  # A is the vector of binding coalitions
  # Si se requiere el juego v_{epsilon}
  vt <- NULL # El juego v_{epsilon}
  if (exists("vt")) {
    vt <- v - t
    vt[nC] <- v[nC]
  }

  if (binary==FALSE) {
    for (S in 1:length(sat)) {
      sat[S] <- codebin2lex(n=n,Nbin=sat[S])
    }
    sat <- sort(sat) # ponemos las coaliciones en orden lexicográfico
    vt <- bin2lex(vt)
  } else {
    sat <- sort(sat) # ponemos las coaliciones en orden binario
  }

  return(list(t = t, sat = sat, x = x, vt = vt))

}
