#' @title Weighted majority game
#' @description This function returns the characteristic function of the described weighted majority game.
#' @param q A quota, as a number between 0 and the sum of player weights.
#' @param w The player weights, as a vector of non-negative numbers.
#' @param binary A logical value. By default, \code{binary=FALSE}.
#' @return The characteristic function of the weighted majority game associated with the described situation, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Given a situation in which a number of agents have to vote for or against a certain measure,
#' let \eqn{N =\{1,\ldots,n\}} be the set of voters, \eqn{w} be a non-negative vector of voter weights (the weight of each voter is the number of votes or the proportion of total votes they hold),
#' and \eqn{q \in [0,\sum_{i \in N}w_{i}]} be the quota (the minimum number of votes or the minimum proportion of total votes needed to pass the measure).
#' The corresponding weighted majority game, \eqn{v}, is defined by
#' \deqn{v(S)=1 \text{ if } \sum_{i \in S}w_{i} \geqslant q \text{ and } v(S)=0 \text{ otherwise, for each }S\in 2^N.}
#' @examples
#' q <- 39
#' w <- c(rep(7,5),rep(1,10))
#' weightedmajoritygame(q,w)
#' @export

weightedmajoritygame=function(q,w,binary=FALSE) {

  ################################
  ###Comprobación datos entrada###
  ################################

  if(sum(w<0)>0){
    stop("'w' must not have negative elements.")
  }
  if(q<=min(w)){
    warning("'q' is not greater than any of the elements in 'w'.")
  }
  if(q>sum(w)){
    warning("'q' is greater than the sum of the elements in 'w'.")
  }

  ################################
  #####Creación de variables######
  ################################

  n=length(w) #Número de jugadores
  nC=2^n-1 #Número de coaliciones
  v=rep(0,nC) #Juego v

  ################################
  ######Cuerpo de la función######
  ################################

  for(S in 1:nC)
  {
    num=S
    J=floor(log2(num))+1 #Jugador más grande
    Sjug=c()#Jugadores en S
    for (ii in 1:J)#Algoritmo de paso de un número en decimal a binario.
    {
      if (num %% 2 ==1)
      {
        Sjug=c(Sjug,ii)
      }
      num=num %/% 2
    }
    if(sum(w[Sjug])>=q)
      v[S]=1
  }

  ################################
  ######Salidas de la función#####
  ################################
  if (binary == FALSE) { # Devolvemos el juego en orden lexicográfico.
    return(bin2lex(v))
  } else { # Devolvemos el juego en orden binario.
    return(v)
  }
}
