#' @title Get permutation
#' @description Given a number of players and a position, this function returns the permutation of players that occupies the given position when permutations are arranged according to the Lehmer code.
#' @param n Number of players, as an integer.
#' @param pos Position according to the Lehmer code, as an integer.
#' @return The permutation of \code{n} players whose Lehmer code position is \code{pos}, as a vector.
#' @details The Lehmer code makes use of the fact that there are \eqn{n!} permutations of a sequence of \eqn{n} numbers.
#' If a permutation \eqn{\sigma} is specified by the sequence \eqn{(\sigma_{i})_{i=1}^{n}}, its Lehmer code is the
#' sequence \eqn{L(\sigma)=(L(\sigma)_{i})_{i=1}^{n}}, where \eqn{L(\sigma)_i=|\{j>i:\sigma_j<\sigma_i\}|}.
#'
#' The position of permutation \eqn{\sigma} according to the Lehmer code order is
#' \deqn{L_{\sigma}=1 + \sum_{i=1}^{n} (n-i)! L(\sigma)_i}.
#' @examples
#' getpermutation(4, 5)
#' n <- 4
#' for (i in 1:factorial(n)) {
#'   cat("[", i, "]", paste(getpermutation(n,i)), "\n")
#' }
#' @seealso \link{getpermutationnumber}
#' @export

getpermutation <- function(n, pos) {

  ##################################
  ### Comprobación datos entrada ###
  ##################################

  if (pos < 0 || pos %% 1 != 0) {
    stop("'pos' must be a positive integer.")
  }

  if (n < 0 || n %% 1 != 0) {
    stop("'n' must be a positive integer.")
  }

  if (pos > factorial(n)) {
    stop("'pos' must be smaller or equal than n!.")
  }

  ################################
  ###### Cuerpo de la función######
  ################################

  # Primero calculamos la codificación de Lehmer de la permutación número pos.  El código de Lehmer
  # cuenta el número de términos en la permutación a la derecha de cada uno que son más pequeños que él.
  pos <- pos - 1
  d <- rep(0, n)
  divisor <- 2 # Hacemos todas las divisiones empezando por 2 hasta encontrar un 0.
  while (pos != 0) {
    d[n - divisor + 1] <- pos %% divisor
    pos <- trunc(pos / divisor)
    divisor <- divisor + 1
  }

  # Hacemos la descodificación como se indica en la página de wikipedia:
  # For decoding a Lehmer code into a permutation of a given set: for each entry x,
  # in order from right to left, correct the items to its right by adding 1 to all
  # those (currently) greater than or equal to x; finally interpret the resulting
  # permutation of {0, 1, … n − 1} as sequence numbers (which amounts to adding 1
  # to each entry if a permutation of {1, 2, … n} is sought).

  permu <- d

  for (ii in (n - 1):1)
  {
    permu[(ii + 1):length(permu)] <- permu[(ii + 1):length(permu)] + as.numeric(permu[ii] <= permu[(ii + 1):length(permu)])
  }

  permu <- permu + 1 # La ordenación parte del 0, por lo que añadimos un 1.
  ################################
  ###### Salidas de la función#####
  ################################

  return(permu)
} # Fin de la función
