#' @title Degenerate check
#' @description This function checks if the given game is degenerate.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @return \code{TRUE} if the game is degenerate, \code{FALSE} otherwise.
#' @details  A game \eqn{v\in G^N} is degenerate if \eqn{v(N)=\sum_{i \in N} v(i)}.
#' @examples
#' v <- c(1, 5, 10, 0, 0, 0, 16)
#' degeneratecheck(v)
#' w <- c(1, 5, 10, 0, 0, 0, 15)
#' degeneratecheck(w)
#' @seealso \link{essentialcheck}
#' @export

degeneratecheck <- function(v, binary = FALSE) {

  # Comprobaciones iniciales.
  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  # Definimos variables que necesitamos.
  checkR <- TRUE # juego degenerado hasta que se demuestre lo contrario
  tol <- 100*.Machine$double.eps

  # Si el juego se introduce en lexicográfico, comprobamos directamente en lexicográfico.
  # Si el juego se introduce en binario, comprobamos directamente en binario.
  if (binary == FALSE) {
    if (abs(v[nC] - sum(v[1:n])) > tol) {
      checkR <- FALSE
    }
  } else {
    if (abs(v[nC] - sum(v[2.^(seq(1, n)-1)])) > tol) {
      checkR <- FALSE
    }
  }

  return(checkR)
} # Fin de la función
