#' @title Coalition-weighted Shapley value
#' @description Given a game and a weight family, this function returns the coalition-weighted Shapley value.
#' @param v A characteristic function, as a vector.
#' @param delta A weight family. It can be introduced in two different ways: as a non-negative vector whose length is the number of coalitions (thus specifying all coalition weights) or as a non-negative vector whose length is the number of players (thus specifying the weights of single-player coalitions and implying that the rest of weights are 0). In any case, if the introduced weights do not add up to 1, the weight family is computed by normalization.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} and \code{delta} (assuming \code{delta} was introduced by specifying all coalition weights; otherwise there is no difference) are introduced in binary order instead of lexicographic order.
#' @param game A logical value. By default, \code{game=FALSE}. If set to \code{TRUE}, the coalition-weighted game is also returned.
#' @return The coalition-weighted Shapley value, as a vector. If \code{game=TRUE}, the coalition-weighted game is also returned, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details A weight family is a collection of \eqn{2^{|N|}-1} real numbers \eqn{\delta=\{\delta_{T}\}_{T \in 2^{N} \setminus \emptyset}} such that \eqn{\delta_{T} \geqslant 0} for each \eqn{T \in 2^{N} \setminus \emptyset} and \eqn{\sum_{T \in 2^{N} \setminus \emptyset}\delta_{T}=1}.
#' For each \eqn{v \in G^{N}} and each \eqn{T \in 2^{N}}, the T-marginal game of \eqn{v}, denoted \eqn{v^{T} \in G^{N}}, is defined as
#' \deqn{v^{T}(S)=v(S \cup (N \setminus T))-v(N \setminus T)+v(S \cap (N \setminus T)), \ S \in 2^{N}.}
#' For each game \eqn{v \in G^{N}} and each weight family \eqn{\delta}, the \eqn{\delta}-weighted game \eqn{v^{\delta} \in G^{N}} is defined as
#' \deqn{v^{\delta} = \sum_{T \in 2^{N} \setminus \emptyset}\delta_{T}v^{T}.}
#' Given a game \eqn{v \in G^{N}}, its \eqn{\delta}-weighted Shapley value, \eqn{\Phi^{\delta}(v)}, is the Shapley value of the \eqn{\delta}-weighted game:
#' \deqn{\Phi^{\delta}(v)=Sh(v^{\delta}).}
#' @examples
#' v <- c(0,0,0,0,0,0,1,0,0,1,3,4,6,8,10)
#' delta <- c(0.3,0.1,0,0.6,0,0,0,0,0,0,0,0,0,0,0)
#' coalitionweightedshapleyvalue(v, delta, binary=TRUE)
#'
#' v <- c(0,0,0,0,0,0,0,0,1,4,1,3,6,8,10)
#' delta <- c(0.25,0.25,0.25,0.25)
#' a <- coalitionweightedshapleyvalue(v, delta, game=TRUE)
#' b <- coalitionweightedshapleyvalue(a$game, delta, game=TRUE)
#' c <- coalitionweightedshapleyvalue(b$game, delta, game=TRUE)
#' plotcoresets(rbind(v, a$game, b$game, c$game), imputations=FALSE)
#'
#' # Games a, b and c have the same Shapley value:
#' all(sapply(list(a$value, b$value, c$value, shapleyvalue(v)),
#'            function(x) all.equal(x, a$value) == TRUE))
#' @seealso \link{marginalgame},  \link{shapleyvalue}, \link{weightedshapleyvalue}
#' @references Sánchez Rodríguez, E., Mirás Calvo, M. A., Quinteiro Sandomingo, C., & Núñez Lugilde, I. (2024). Coalition-weighted Shapley values. \emph{International Journal of Game Theory}, 53, 547-577.
#' @export

coalitionweightedshapleyvalue <- function(v, delta, binary = FALSE, game = FALSE) {

  #En esta función, primero calculamos el juego ponderado y después
  #el valor de Shapley de dicho juego

  ################################
  ###Comprobación datos entrada###
  ################################

  nC=length(v) # número de coaliciones
  n=log(nC+1)/log(2) # número de jugadores

  if (n>floor(n))
  {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if(length(delta)!=nC & length(delta)!=n)
  {
    stop("The length of 'delta' must be either the length of 'v' or the number of players in 'v'.")
  }

  for (S in 1:length(delta)) {
    if (delta[S]<0) {
      stop("'delta' must be a vector of non-negative numbers.")
    }
  }

  if (sum(delta)==0) {
    stop("The sum of the numbers in 'delta' must be positive.")
  }

  if (binary == FALSE) { # Si v se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  if(length(delta)==n) # Si en delta se especifican solo los pesos de las coaliciones individuales, fabricamos el vector con todos los pesos, ya en orden binario.
  {
    delta.reducido <- delta
    delta<-numeric(nC)
    for (i in 1:n) {
      delta[2^(i-1)]<-delta.reducido[i]
    }
  } else { # Si en delta se especifican los pesos de todas las coaliciones...
    if (binary == FALSE) { # Si delta se introdujo en lexicográfico, lo pasamos a binario.
      delta <- lex2bin(delta)
    }
  }
  # Ahora ya tenemos v y delta en orden binario.

  if(sum(delta)!=1)
  {#Normalizamos el vector de pesos.
    warning('In coalitionweightedshapleyvalue(v,delta), the elements of delta do not add up to 1. Therefore coalitional weights are being normalized.',call.=F)
    delta=delta/sum(delta)
  }

  ################################
  #####Creación de variables######
  ################################

  #Inicializamos el delta-weighted game:
  vdelta=rep(0,nC)

  ################################
  ######Cuerpo de la función######
  ################################

  #Calculamos el delta-weighted game:
  #Primero nos deshacemos de todas las coaliciones con pesos iguales a cero.

  for(S in which(delta!=0))
  {#El juego ponderado será la suma ponderada de los marginales con los pesos de cada coalición.
    vdelta=vdelta+delta[S]*marginalgame(v,S,binary=TRUE)
  }

  #Por último, calculamos el valor de Shapley del juego ponderado.
  CWS=shapleyvalue(vdelta,binary=TRUE)

  ################################
  ######Salidas de la función#####
  ################################
  if (game == FALSE) { # Si el usuario no pide vdelta...
    return(CWS)
  } else { # Si el usuario pide vdelta...
    if (binary == FALSE) { # Devolvemos vdelta en orden lexicográfico.
      return(list(value=CWS,game=bin2lex(vdelta)))
    } else { # Devolvemos vdelta en orden binario.
      return(list(value=CWS,game=vdelta))
    }
  }

}# Fin de la función
