% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/learn_allocation_rule.R
\name{learn_allocation_rule}
\alias{learn_allocation_rule}
\title{Build an Optimal Adaptive Allocation Rule using Reinforcement Learning}
\usage{
learn_allocation_rule(
  models,
  N_total,
  N_ini,
  N_block,
  Delta,
  outcome_type = c("continuous", "binary"),
  sd_normal = NULL,
  optimization_metric = c("MAE", "power", "TD", "power and MAE"),
  rl_models = models,
  rl_models_prior = NULL,
  seed = NULL,
  rl_config = rl_config_set(),
  alpha = 0.025,
  selModel = c("AIC", "maxT", "aveAIC"),
  Delta_range = c(0.9, 1.1) * Delta,
  output_dir = format(Sys.time(), "\%Y\%m\%d_\%H\%M\%S"),
  output_base_dir = "allocation_rules",
  checkpoint_dir = "checkpoints"
)
}
\arguments{
\item{models}{An object of class \link[DoseFinding]{Mods} specifying assumed
dose-response models. When \code{outcome_type} is "binary", \code{models} should
be specified on the logit scale.}

\item{N_total}{A positive integer value. The total number of subjects.}

\item{N_ini}{A positive integer vector in which each element is greater than
or equal to 2. The number of subjects initially assigned to each dose.}

\item{N_block}{A positive integer value. The number of subjects allocated
adaptively in each round.}

\item{Delta}{A positive numeric value. The clinically relevant target effect.
When \code{outcome_type} is "binary", \code{Delta} should be specified
on the logit scale. See \link[DoseFinding]{TD} for details.}

\item{outcome_type}{A character value specifying the outcome type.
Possible values are "continuous" (default), and "binary".}

\item{sd_normal}{A positive numeric value. The standard deviation of the
observation noise. When \code{outcome_type} is "continuous",
\code{sd_normal} must be specified.}

\item{optimization_metric}{A character value specifying the metric to
optimize. Possible values are "MAE" (default), "power", "TD", or
"power and MAE". See Section 2.2 of the original paper for details.
"power and MAE" shows performance between "power" and "MAE"
by setting the reward based on MAE to 0 when not significant.}

\item{rl_models}{An object of class \link[DoseFinding]{Mods}. True dose-response
models in simulations for reinforcement learning. The default is the
same as the 'models' argument. Empirically, the inclusion of a wide
variety of models tends to stabilize performance (See RL-MAE incl. exp
in the supporting information of the original paper).}

\item{rl_models_prior}{A positive numeric vector. The probability or weight
with which each model in rl_models is selected as the true model in
the simulation. The default is NULL, which specifies equal probability
for each model.}

\item{seed}{An integer value. Random seed for reinforcement learning.}

\item{rl_config}{A list. Other settings for reinforcement learning. See
\link{rl_config_set} for details.}

\item{alpha}{A positive numeric value. The significance level. Default is 0.025.}

\item{selModel}{A character value specifying the model selection criterion
for dose estimation. Possible values are "AIC" (default), "maxT", or
"aveAIC". See \link[DoseFinding]{MCPMod} for details.}

\item{Delta_range}{A numeric vector of length 2. The lower and upper bounds
of Delta where the estimated target dose is correct. Default is
\code{c(0.9, 1.1) * Delta}.}

\item{output_dir}{A character value. Directory name or path to store the
built allocation rule. Default is the current datetime.}

\item{output_base_dir}{A character value. Parent directory path where the
built allocation rule will be stored. Valid only if 'output_dir' does
not contain '/'. Default is "allocation_rules".}

\item{checkpoint_dir}{A character value. Parent directory path to save
checkpoints. It enables you to resume learning from that point onwards.
Default is "checkpoints".}
}
\value{
An \link{AllocationRule} object.
}
\description{
Build an Optimal Adaptive Allocation Rule using Reinforcement Learning
}
\examples{
library(RLoptimal)

doses <- c(0, 2, 4, 6, 8)

# We build the dose-response models to be used in the MCPMod method, 
# which we plan to execute at the end of the clinical trial.
models <- DoseFinding::Mods(
  doses = doses, maxEff = 1.65,
  linear = NULL, emax = 0.79, sigEmax = c(4, 5)
)

# We obtain an optimal adaptive allocation rule by executing 
# `learn_allocation_rule()` with the `models`.
\dontrun{
allocation_rule <- learn_allocation_rule(
  models,
  N_total = 150, N_ini = rep(10, 5), N_block = 10, Delta = 1.3,
  outcome_type = "continuous", sd_normal = sqrt(4.5), 
  seed = 123, rl_config = rl_config_set(iter = 1000),
  alpha = 0.025
)}

# It is recommended that the models used in reinforcement learning include 
# possible models in addition to the models used in the MCPMod method. 
# Here, we add the exponential model according to the supporting information 
# in the original paper.
rl_models <- DoseFinding::Mods(
  doses = doses, maxEff = 1.65,
  linear = NULL, emax = 0.79, sigEmax = c(4, 5), exponential = 1
)

# Then, we specify the argument `rl_models` in `learn_allocation_rule` function.
\dontrun{
allocation_rule <- learn_allocation_rule(
  models,
  N_total = 150, N_ini = rep(10, 5), N_block = 10, Delta = 1.3,
  outcome_type = "continuous", sd_normal = sqrt(4.5), 
  seed = 123, rl_models = rl_models, rl_config = rl_config_set(iter = 1000),
  alpha = 0.025
)}

}
