% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MuFiMeshGP.R
\name{MuFiMeshGP}
\alias{MuFiMeshGP}
\title{Prediction of the MuFiMeshGP emulator for any fidelity level.}
\usage{
MuFiMeshGP(
  X,
  t,
  Y,
  covtype = "Gaussian",
  trend.type = "OK",
  trend.dim = "input",
  trend.pol = "quadratic",
  interaction = NULL,
  mean.known = NULL,
  H.known = NULL,
  gradient = TRUE,
  init = NULL,
  single_fidelity = FALSE,
  param.bounds = NULL,
  iso = FALSE,
  l = 4,
  nugget = 1e-06,
  ncores = 1
)
}
\arguments{
\item{X}{matrix of input locations. Each row represents a sample.}

\item{t}{vector of fidelity levels. Each element is a sample and is connected
to the corresponding row in \code{X}.}

\item{Y}{vector of response values.}

\item{covtype}{covariance kernel type, only 'Gaussian' is available for now,
'Matern5_2' or 'Matern3_2' will be available soon (see \code{\link{cov_gen}}).}

\item{trend.type, trend.dim, trend.pol, interaction}{define the mean function form of
the Gaussian process. \code{trend.type} can be: "SK" in which case
\code{mean.known} needs to be specified as a scalar; "OK" in which case the constant
mean will be evaluated through MLE; "UK" in which case \code{trend.dim}
specifies whether the trend will be along the input space (\code{"input"}),
the fidelity space (\code{"fidelity"}), or both (\code{"both"}).
If \code{trend.dim} is \code{"input"} or \code{"both"}, the user can use the
\code{trend.pol} to specify if the trend on the input space alone should be
\code{"linear"} or \code{"quadratic}.
Finally, if \code{trend.dim} is \code{"both"}, then an \code{interaction}
term specify the polynomial order (\code{"linear} or \code{"quadratic"}) of
 the input space trend that is multiplied to the fidelity space trend.
 See \code{\link{regF_gen}} for further details.}

\item{mean.known}{Specifies the mean if \code{"SK"} as \code{trend.type},
scalar.}

\item{H.known}{allow the user to specify the value of H as
\code{H.known}, a scalar in (0,1).}

\item{gradient}{whether or not the gradient of the log-likelihood shouldbe
used in the parameter estimation.}

\item{init}{Where should the parameter estimation start from, a vector.}

\item{single_fidelity}{can be used as \code{TRUE} to use \code{MuFiMeshGP}
as a single fidelity Gaussian Process. This will set \code{sigma2sq} as 0.}

\item{param.bounds}{a list with two arguments(\code{lower} and \code{upper})
 describing the bounds used for MLE optimization of \code{phi1sq} and \code{phi2sq}.
Each argument should be a vector of length \code{ncol(X)}.
If \code{NULL} the bounds of \code{phi1sq} and \code{phi2sq}
are specified automatically from the design matrix.}

\item{iso}{whether the covariance function will be isotropic (\code{TRUE}
or \code{FALSE})}

\item{l}{rate of convergence of the system (see Details), scalar.}

\item{nugget}{(optional) for controlling numerical error.}

\item{ncores}{(optional) number of cores for parallelization.}
}
\value{
a list which is given the S3 class "MuFiMeshGP"
}
\description{
The function computes the posterior mean and standard deviation of the
MuFiMeshGP model.
}
\details{
From the model fitted by \code{\link{MuFiMeshGP}} or \code{\link{update.MuFiMeshGP}}
 the posterior mean and standard deviation are calculated for any input
 location and fidelity level.
 For details, see Boutelet and Sung (2025, <arXiv:2503.23158>).
}
\examples{
# Example code

f <- function(x, t){
  x <- c(x)
  return(exp(-1.4*x)*cos(3.5*pi*x)+sin(40*x)/10*t^2)
}

set.seed(1)
X <- matrix(runif(15,0,1), ncol = 1)
tt <- runif(15,0.5,2)

Y <- f(c(X), tt)

fit.mufimeshgp <- MuFiMeshGP(X, tt, Y)

xx <- matrix(seq(0,1,0.01), ncol = 1)
ftrue <- f(xx, 0)

# predict
pred.mufimeshgp <- predict(fit.mufimeshgp, xx, rep(0,101))

mu <- pred.mufimeshgp$mean
s <- pred.mufimeshgp$sd
lower <- mu + qnorm(0.025)*s
upper <- mu + qnorm(0.975)*s

# plot

oldpar <- par(mfrow = c(1,1))
plot(xx, ftrue, "l", ylim = c(-1,1.3), ylab = "y", xlab = "x")
lines(c(xx), mu, col = "blue")
lines(c(xx), lower, col = "blue", lty = 2)
lines(c(xx), upper, col = "blue", lty = 2)
points(c(X), Y, col = "red")
par(oldpar)

### RMSE ###
print(sqrt(mean((ftrue - mu))^2))

}
\seealso{
\code{\link{MuFiMeshGP}} for the model.
}
