% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DispFieldST.R
\name{DispFieldST}
\alias{DispFieldST}
\title{Displacement fields for spatiotemporal data when velocity is spatially constant}
\usage{
DispFieldST(inputstack1, lag1, factv1, facth1, restricted = FALSE)
}
\arguments{
\item{inputstack1}{a raster stack with each raster layer representing an
instance of time. The raster stack should be organized such that the first
raster in the stack is the first observed spatial dataset and time
progresses forward with the third dimension index of the raster stack. The
raster stack should contain only numeric values. Any NA value will be
converted to a zero}

\item{lag1}{an integer time lag}

\item{factv1}{an odd integer for the vertical dimension of subgrids}

\item{facth1}{an odd integer for the horizontal dimension of subgrids}

\item{restricted}{logical (TRUE or FALSE)}
}
\value{
A data frame is returned with the following column names: rowcent,
colcent, frowmin, frowmax, fcolmin, fcolmax, centx, centy, dispx, and
dispy. The rowcent and colcent column names are the row and column indices
for the center of each sub-grid; frowmin and frowmax are the sub-grid
minimum and maximum row indices; fcolmin and fcolmax are the sub-grid
minimum and maximum column indices; centx and centy are the projected
coordinates of the centre of the subgrid derived from the raster input
files; dispx and dispy are the orthoganal velocity vectors in units of
space per timestep in the horizontal and vertical directions in the same
spatial units as the projected coordinates of the raster input files.
}
\description{
This is an implementation of a novel algorithm that differs from more
traditional digital image correlation implementations that are applied in the
\code{\link{DispField}} and \code{\link{DispFieldbb}} functions. The function
calculates a displacement field representing persistent movement based on the
cross-covariance in a raster stack (in this case a sequential series of
rasters) presumably representing spatial population abundance or density at
more than two different instances of time. If analysis is restricted to only
two time instances, \code{\link{DispField}} is more appropriate.
}
\details{
The input rasters in the raster stack are first converted to equivalent
matrices, which together represent a three-dimensional array with two spatial
dimensions and one time dimension. The prescribed lag is applied to the three
dimensional array derived from the raster stack by first producing two
equivalent arrays and then removing appropriate numbers of layers from the
top of one and the bottom of the other. These are referred to as unlagged and
lagged spatiotemporal arrays in the description that follows.

Prior to computing displacement based on direction of maximum
cross-covariance, the function divides the spatial domain up into sub-grids
of size factv1 X facth1, which are vertical and horizontal sub-grid spatial
dimensions.

The function converts three dimensional lagged and unlagged spatiotemporal
arrays to two-dimensional lagged and unlagged spatiotemporal matrices by
averaging along one of the spatial dimensions (either rows or columns) to
obtain two pairs of two-dimensional matrices in which one dimension is
spatial (either rows or columns) and one dimension is temporal. One of each
pair corresponds to the  unlagged spatiotemporal array and the other
corresponds to the lagged spatiotemporal array. Displacement in the vertical
direction is computed using unlagged and lagged matrices that have been
averaged along rows and displacement in the horizontal direction is computed
using unlagged and lagged matrices that have been averaged along columns.

If restricted is set to FALSE (the default), the function computes
cross-covariance between each sub-grid of the unlagged row-averaged
spatiotemporal matrix and the whole row-averaged lagged spatiotemporal matrix
and between each sub-grid of the unlagged column-averaged spatiotemporal
matrix and the entirety corresponding lagged matrix.

If restricted is set to TRUE, the function uses cross-covariance between
lagged and unlagged version of row-averaged and column averaged
spatiotemporal matrices that have all been either row or column-averaged
within sub-grids to estimate vertical and horizontal displacement.

Regardless of whether restricted is set to TRUE or FALSE, for each sub-grid,
displacement in the x and y direction is divided by the shift in the time
dimension to produce orthogonal velocity vetors. Note that for this reason,
the lag1 argument of the function does not necessarily determine the time lag
that is used to produce each orthoganal velocity vector.

Reference coordinates and cell size are extracted from the first raster stack
such that the locations from whence displacement is estimated as well as
displacement (or velocity) estimates can be expressed in the units of the
projected coordinates.

The coordinates are assumed to increase vertically and horizontally from the
lower left corner of the two-dimensional domain.

Caution is warranted when defining the sub-grid dimensions because the
function can produce erroneous results when sub-grids are too small.

In addition, results can be quite sensitive to specification of the time lag.
If velocities are highly variable in space or over time, avoid specifying a
single time lag by calling the related \code{\link{DispFieldSTall}} function.
}
\examples{
(Mat1 <- matrix(rep(c(1:5, 0, 0, 0, 0), 9), nrow = 9, byrow = TRUE))
(Mat2 <- matrix(rep(c(0, 1:5, 0, 0, 0), 9), nrow = 9, byrow = TRUE))
(Mat3 <- matrix(rep(c(0, 0, 1:5, 0, 0), 9), nrow = 9, byrow = TRUE))
(Mat4 <- matrix(rep(c(0, 0, 0, 1:5, 0), 9), nrow = 9, byrow = TRUE))

# rasterizing
rast1 <- terra::rast(Mat1)
terra::plot(rast1)
rast2 <- terra::rast(Mat2)
terra::plot(rast2)
rast3 <- terra::rast(Mat3)
terra::plot(rast3)
rast4 <- terra::rast(Mat4)
terra::plot(rast4)

teststack1 <- c(rast1, rast2, rast3, rast4)
(VFdf2 <- DispFieldST(teststack1, lag1 = 1, factv1 = 9, facth1 = 9))
# block is moving rightward at a speed of 1 unit of space per unit of time
# dispx = 1
}
\seealso{
\code{\link{DispField}} for a similar function with a grid of focal
regions for only two time instances, \code{\link{DispFieldSTbb}} for a
version designed to quantify persistent directional movement when the time
series features more than two time instances but using a bounding pox to
define a focal region, see \code{\link{DispFieldSTall}} for a version
designed to quantify persistent directional movement when velocity is
variable in space, and \code{\link{Xcov2D}} for demonstration of how
two-dimensional cross-covariance is used to determine displacement (see
examples of Xcov2D function documentation).
}
