#' Screen wear and extract the longest valid minute-level segment
#'
#' Determines the native epoch from the first two time stamps, decimates to
#' 1-minute resolution (if needed), checks total minutes \eqn{\ge} `1440*min_days`,
#' and within that finds the **longest** contiguous segment where runs of
#' consecutive zeros do not exceed `max_zero_run` minutes.
#'
#' @param df `data.frame` with columns `Date`, `Time`, `Activity` in time order.
#' @param min_days Integer minimum number of whole days required (default 5).
#' @param max_zero_run Integer maximum allowed length (minutes) of a run of zeros.
#' @param date_col Name of the date column.
#' @param time_col Name of the time column.
#' @param activity_col Name of the activity column used to determine wear/non-wear.
#' @return A list with elements:
#' \describe{
#'   \item{status}{`"ok"` if a qualifying segment is found, otherwise a message.}
#'   \item{epoch_inferred}{Detected original epoch in seconds (15, 30, 60).}
#'   \item{out_idx}{Indices of the kept segment in the input.}
#'   \item{clean_df}{Minute-level `data.frame` of the selected segment (if ok).}
#' }
#'
#' @details
#' Decimation rule: 15 s \eqn{\rightarrow} factor 4; 30 s \eqn{\rightarrow} factor 2;
#' 60 s \eqn{\rightarrow} factor 1. The zero-run criterion is applied on the
#' 1-minute `Activity` series.
#'
#' @seealso [import_acti_file()], [sleep_cos()], [sleep_detection()]
#'
#' @export


screen_wear <- function(
    df,
    min_days     = 5L,    # N days
    max_zero_run = 120L,  # n minutes of consecutive zeros
    date_col     = "Date",
    time_col     = "Time",
    activity_col = "Activity"
){
  stopifnot(all(c(date_col, time_col, activity_col) %in% names(df)))
  if (!is.numeric(df[[activity_col]])) stop("Activity must be numeric.")

  # --- parse Time -> seconds since midnight (supports HH:MM[:SS], AM/PM, fraction-of-day, or seconds) ---
  parse_time_sec <- function(x){
    if (is.numeric(x)) {
      x_ok <- x[is.finite(x)]
      if (!length(x_ok)) stop("Time column has no finite numeric values.")
      minx <- min(x_ok); maxx <- max(x_ok)
      if (minx >= 0 && maxx <= 1.1)  return(round(x * 86400))   # fraction of day
      if (minx >= 0 && maxx <= 86401) return(round(x))          # seconds since midnight
      stop("Numeric Time must be fraction-of-day [0..1] or seconds [0..86400].")
    }
    x <- trimws(as.character(x))
    is_pm <- grepl("(?i)PM", x); is_am <- grepl("(?i)AM", x)
    z <- gsub("(?i)\\s*(AM|PM)\\s*", "", x)
    parts <- strsplit(z, ":", fixed = TRUE)
    conv <- function(p){
      h <- as.integer(p[1]); m <- as.integer(ifelse(length(p)>=2, p[2], 0))
      s <- as.integer(ifelse(length(p)>=3, p[3], 0))
      h <- ifelse(is.na(h), 0L, h); m <- ifelse(is.na(m), 0L, m); s <- ifelse(is.na(s), 0L, s)
      h*3600L + m*60L + s
    }
    sec <- vapply(parts, conv, integer(1))
    hrs <- sec %/% 3600L
    pm_add <- ifelse(is_pm & hrs < 12L, 12L*3600L, 0L)   # 1–11 PM -> +12h
    am_fix <- ifelse(is_am & hrs == 12L, -12L*3600L, 0L) # 12 AM -> 0h
    sec + pm_add + am_fix
  }

  # --- build absolute timeline in MINUTES (no calendar parsing) ---
  dvec <- as.character(df[[date_col]])
  if (anyNA(dvec)) stop("Date has NA; remove/clean missing Date rows.")

  # Increase day index when Date string changes in row order
  day_idx <- c(0L, cumsum(dvec[-1] != dvec[-length(dvec)]))
  tsec    <- parse_time_sec(df[[time_col]])
  abs_sec <- as.numeric(day_idx) * 86400 + as.numeric(tsec)
  abs_min <- abs_sec %/% 60L

  ok <- is.finite(abs_sec) & is.finite(df[[activity_col]])
  abs_sec <- abs_sec[ok]; abs_min <- abs_min[ok]
  Activity <- df[[activity_col]][ok]
  Date_keep <- dvec[ok]
  Time_keep <- as.character(df[[time_col]][ok])
  id_keep   <- if ("id" %in% names(df)) df$id[ok] else NA_character_

  # order & combine duplicates
  o <- order(abs_sec)
  abs_sec <- abs_sec[o]; abs_min <- abs_min[o]
  Activity <- Activity[o]; Date_keep <- Date_keep[o]; Time_keep <- Time_keep[o]; id_keep <- id_keep[o]

  r_ts <- rle(abs_sec)
  end  <- cumsum(r_ts$lengths)
  start<- end - r_ts$lengths + 1L
  agg_sec   <- r_ts$values
  agg_min   <- abs_min[start]
  agg_act   <- mapply(function(a,b) sum(Activity[a:b], na.rm = TRUE), start, end)
  agg_date  <- Date_keep[start]
  agg_time  <- Time_keep[start]
  agg_id    <- id_keep[start]

  # --- infer epoch (seconds) from first two unique timestamps ---
  if (length(agg_sec) < 2) stop("Not enough unique timestamps to infer epoch.")
  epoch_sec <- as.integer(round(agg_sec[2] - agg_sec[1]))
  if (!epoch_sec %in% c(15L, 30L, 60L))
    stop(sprintf("Epoch must be 15, 30, or 60 seconds; detected %d.", epoch_sec))

  # --- decimate to 1-min: 15s->/4, 30s->/2, 60s->/1 ---
  f <- switch(as.character(epoch_sec), "60"=1L, "30"=2L, "15"=4L)
  keep_idx <- seq(1L, length(agg_sec), by = f)

  dec_min  <- agg_min[keep_idx]     # minute index (1-min grid)
  dec_act  <- agg_act[keep_idx]
  dec_date <- agg_date[keep_idx]
  dec_time <- agg_time[keep_idx]
  dec_id   <- agg_id[keep_idx]

  # --- early screen: total minutes must be >= 1440 * min_days ---
  total_minutes <- length(dec_min)
  need_minutes  <- 1440L * as.integer(min_days)
  if (total_minutes < need_minutes) {
    return(list(
      status           = "screened_out_insufficient_total_minutes",
      epoch_inferred   = epoch_sec,
      decimation_factor= f,
      total_minutes    = total_minutes,
      required_minutes = need_minutes,
      clean_df         = data.frame()
    ))
  }

  # --- break on zero-runs >= max_zero_run, then enforce min-days on segments ---
  is_zero <- is.finite(dec_act) & dec_act == 0
  r <- rle(is_zero)
  r_ends   <- cumsum(r$lengths)
  r_starts <- r_ends - r$lengths + 1L
  long_zero <- which(r$lengths[r$values==1] >= (max_zero_run))

  # split at end of each long zero run
  breaks <- rep(FALSE, length(dec_act))
  if (length(long_zero)) breaks[r_ends[long_zero]] <- TRUE
  seg_id <- cumsum(breaks) + 1L

  # segment summaries (days via minute index; 1 day = 1440 mins)
  day_idx_min <- dec_min %/% 1440L
  seg_df <- do.call(rbind, lapply(split(seq_along(dec_min), seg_id), function(ix){
    i1 <- ix[1]; i2 <- ix[length(ix)]
    mins_span <- i2 - i1 + 1L
    days_span <- (day_idx_min[i2] - day_idx_min[i1]) + 1L
    rz <- rle(is_zero[ix]); mx0 <- if (!any(rz$values)) 0L else max(rz$lengths[rz$values])
    data.frame(i1=i1, i2=i2, mins_span=mins_span, days_span=days_span,
               max_zero_run_obs=mx0,
               qualifies = (days_span >= as.integer(min_days) &
                              mx0 < as.integer(max_zero_run)))
  }))

  # if no segment satisfies the minimum-day requirement -> SCREEN OUT
  if (!any(seg_df$qualifies)) {
    return(list(
      status           = "screened_out_no_qualifying_segment",
      epoch_inferred   = epoch_sec,
      decimation_factor= f,
      total_minutes    = total_minutes,
      segments         = seg_df,   # for inspection
      clean_df         = data.frame()
    ))
  }

  # keep the longest qualifying segment (by days, then minutes)
  cand <- seg_df[seg_df$qualifies, , drop = FALSE]
  j <- order(cand$days_span, cand$mins_span, decreasing = TRUE)[1]
  chosen <- cand[j, ]

  out_idx <- chosen$i1:chosen$i2
  clean_df <- data.frame(
    id       = dec_id[out_idx],
    Date     = dec_date[out_idx],
    Time     = dec_time[out_idx],
    Activity = dec_act[out_idx],
    stringsAsFactors = FALSE
  )

  list(
    status            = "ok",
    epoch_inferred    = epoch_sec,
    out_idx           = out_idx,
    clean_df          = clean_df
  )
}
